<?php

namespace NioModules\BasicKYC\Controllers;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserMeta;
use App\Traits\WrapInTransaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use NioModules\BasicKYC\BasicKYCModule;
use NioModules\BasicKYC\Helpers\KycDocStatus;
use NioModules\BasicKYC\Helpers\KycFilter;
use NioModules\BasicKYC\Helpers\KycSessionStatus;
use NioModules\BasicKYC\Helpers\KycStatus;
use NioModules\BasicKYC\Models\KycApplicants;
use NioModules\BasicKYC\Models\KycDocs;
use NioModules\BasicKYC\Models\KycSessions;

class AdminKycController extends Controller
{
    private $module;
    private $pathTemp = 'kyc-temp';
    private $pathSave = 'kyc';

    public function __construct(BasicKYCModule $module)
    {
        $this->module = $module;
    }

    public function index(Request $request, $state = null)
    {
        if (is_null($state) || $state === 'process') {
            $whereIn = [KycSessionStatus::COMPLETED];
            $listing = 'process';
        } elseif ($state === 'all') {
            $whereIn = [KycSessionStatus::COMPLETED, KycSessionStatus::APPROVED, KycSessionStatus::REJECTED];
            $listing = 'all';
        } else {
            $whereIn = $state === 'new' ? [KycSessionStatus::STARTED] : [$state];
            $listing = $state;
        }

        $filter = new KycFilter($request);
        $totalCount = KycSessions::whereNotIn('status', [KycSessionStatus::STARTED])->count();
        $sessionQuery = KycSessions::whereIn('kyc_sessions.status', $whereIn)
            ->orderBy('kyc_sessions.id', user_meta('kyc_order', 'desc'));

        $applicants = $sessionQuery->filter($filter)->paginate(user_meta('kyc_perpage', '10'));

        return view('BasicKYC::admin.list', compact('applicants', 'totalCount', 'listing'));
    }

    public function viewSubmission(Request $request, $session)
    {
        $getSession = KycSessions::with(['documents'])->where('session', $session)->firstOrFail();
        $getDocuments = !blank($getSession->documents) ? $getSession->documents : null;
        $this->deleteDuplicateFiles($getSession);

        return view('BasicKYC::admin.details', [
            'data' => $getSession,
            'documents' => $getDocuments,
        ]);
    }

public function updateStatus(Request $request, $session)
{
    \Log::info("KYC Status Update Request", [
        'admin_id' => auth()->user()->id,
        'session_id' => $session,
        'action' => $request->action
    ]);

    if (BasicKYCModule::isInvalid()) {
        \Log::warning("BasicKYCModule marked as invalid. Blocking update.", [
            'admin_id' => auth()->user()->id
        ]);
        throw ValidationException::withMessages(['invalid' => __('Sorry, we are unable to proceed your request.')]);
    }

    $action = $request->action;
    $getSession = KycSessions::find($session);

    if (empty($action) || !in_array($action, ['reject', 'resubmit', 'verify']) || blank($getSession)) {
        \Log::warning("Invalid action or session not found", [
            'action' => $action,
            'session' => $getSession
        ]);
        throw ValidationException::withMessages(['error' => __('Something went wrong. Please reload the page and try again.')]);
    }

    if (data_get($getSession, 'status') === KycSessionStatus::COMPLETED) {
        $applicant = KycApplicants::where('user_id', $getSession->user_id)->first();

        if (blank($applicant) || data_get($applicant, 'status') === KycStatus::STARTED) {
            \Log::warning("Applicant not found or in STARTED status", [
                'session_id' => $getSession->id,
                'user_id' => $getSession->user_id
            ]);
            throw ValidationException::withMessages(['error' => __('Sorry, we are unable to proceed your request.')]);
        }

        if (in_array($applicant->status, [KycStatus::VERIFIED, KycStatus::REJECTED])) {
            \Log::info("Applicant already verified or rejected", [
                'applicant_status' => $applicant->status
            ]);
            throw ValidationException::withMessages([
                'warning' => __('The documents of applicant is already :status.', ['status' => __($applicant->status)])
            ]);
        }

        if ($applicant->status === KycStatus::RESUBMIT) {
            \Log::info("Applicant is waiting for resubmission");
            throw ValidationException::withMessages([
                'warning' => __('The documents of applicant is already rejected & waiting for new submission.')
            ]);
        }

        try {
            return $this->wrapInTransaction(function () use ($action, $getSession, $applicant) {
                $checkedBy = ['id' => auth()->user()->id, 'name' => auth()->user()->name];
                $kycMeta = $action === 'verify' ? 'verified' : $action;

                \Log::info("Updating user meta and session", [
                    'action' => $action,
                    'user_id' => $getSession->user_id,
                    'kyc_meta' => $kycMeta
                ]);

                UserMeta::updateOrCreate([
                    'user_id' => $getSession->user_id,
                    'meta_key' => 'kyc_verification'
                ], [
                    'meta_value' => $kycMeta
                ]);

                if ($action === 'verify') {
                    $this->updateProfileInfo($getSession->user_id, $getSession);

                    UserMeta::updateOrCreate([
                        'user_id' => $getSession->user_id,
                        'meta_key' => 'kyc_verified_at'
                    ], [
                        'meta_value' => Carbon::now()
                    ]);

                    KycDocs::where('session_id', $getSession->id)->update(['state' => KycDocStatus::VALID]);

                    $getSession->update([
                        'status' => KycSessionStatus::APPROVED,
                        'checked_by' => $checkedBy,
                        'checked_at' => Carbon::now()
                    ]);

                    $applicant->update(['status' => KycStatus::VERIFIED]);

                    \Log::info("Applicant verified successfully", ['user_id' => $getSession->user_id]);

                    return response()->json([
                        'type' => 'success',
                        'reload' => true,
                        'title' => __('Verified the Applicant'),
                        'msg' => __('The documents of applicant has been successfully verified.')
                    ]);
                }

                if (in_array($action, ['resubmit', 'reject'])) {
                    KycDocs::where('session_id', $getSession->id)->update(['state' => KycDocStatus::INVALID]);

                    $getSession->update([
                        'status' => KycSessionStatus::REJECTED,
                        'checked_by' => $checkedBy,
                        'checked_at' => Carbon::now()
                    ]);

                    $applicant->update(['status' => $action === 'reject' ? KycStatus::REJECTED : KycStatus::RESUBMIT]);

                    $title = $action === 'reject' ? __('Rejected the Submission') : __('Requested for Resubmission');
                    $message = $action === 'reject' ? __('The documents has been rejected permanently.') : __('The documents has been updated and requested for submit.');

                    \Log::info("Applicant {$action} processed", [
                        'user_id' => $getSession->user_id
                    ]);

                    return response()->json([
                        'type' => 'success',
                        'reload' => true,
                        'title' => $title,
                        'msg' => $message
                    ]);
                }
            });
        } catch (\Exception $e) {
            save_error_log($e);
            \Log::error("Exception while updating KYC status", [
                'error' => $e->getMessage(),
                'session_id' => $session,
                'admin_id' => auth()->user()->id
            ]);
            throw ValidationException::withMessages(['error' => __('Sorry, we are unable to proceed your request.')]);
        }
    }

    \Log::warning("Unexpected request state or session not completed", [
        'session_status' => $getSession->status ?? 'null',
        'action' => $action
    ]);

    throw ValidationException::withMessages(['error' => __('Something went wrong. Please reload the page and try again.')]);
}


    public function downloadDocFile($part, Request $request)
    {
        $id = $request->get('file', 0);

        if ($id) {
            $docs = KycDocs::find(get_hash($id));

            if (!blank($docs)) {
                $docFile = data_get($docs, 'files.' . $part);
                $docType = data_get($docs, 'type');
                $docName = str_replace($part, $docType . '_' . $part, $docFile);

                if (!empty($docFile)) {
                    $pathFile = $this->pathSave . '/' . $docFile;
                    if (Storage::exists($pathFile)) {
                        return Storage::download($pathFile, $docName);
                    }
                }
            }
        }

        return abort(404);
    }

    public function removeDocFiles(array $files)
    {
        foreach ($files as $file) {
            $tempPath = $this->pathTemp . '/' . $file;
            if (Storage::exists($tempPath)) {
                Storage::delete($tempPath);
            }
        }
    }

    private function deleteDuplicateFiles(KycSessions $session)
    {
        try {
            if ($session->status === KycSessionStatus::COMPLETED) {
                $documents = $session->documents()->get();
                foreach ($documents as $document) {
                    $files = data_get($document, 'files', []);
                    $this->removeDocFiles($files);
                }
            }
        } catch (\Exception $e) {
            // Silent fail
        }
    }

    private function updateProfileInfo($userId, $session)
    {
        $user = User::findOrFail($userId);
        $basic = array_merge($user->meta('basics'), $user->meta('addresses'));
        $profile = $session->profile;

        if (!blank($user)) {
            $nameUpdate = false;
            if (data_get($profile, 'name') && $user->name !== data_get($profile, 'name')) {
                $nameUpdate = true;
                $user->name = strip_tags(data_get($profile, 'name'));
                $user->save();
            }

            $metaData = Arr::except($profile, 'name');
            $metaUpdate = false;

            foreach ($metaData as $key => $value) {
                $value = strip_tags($value);
                if (!empty($value) && $value !== $user->meta('profile_' . $key)) {
                    UserMeta::updateOrCreate([
                        'user_id' => $user->id,
                        'meta_key' => 'profile_' . $key
                    ], [
                        'meta_value' => $value
                    ]);
                    $metaUpdate = true;
                }
            }

            if ($nameUpdate || $metaUpdate) {
                $session->revised = $basic;
                $session->save();
            }
        }
    }
}
