<?php

namespace App\Services\Apis\ExchangeRate;

use App\Models\Setting;
use App\Services\Apis\BaseAPi;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpFoundation\Response;

class ExchangeRateApi extends BaseAPi
{
    private $baseUrl = 'https://api.exchangeratesapi.io/v1/latest';
    private $timeOut = 60;

    private function currencies()
    {
        $allowedCurrencies = array_intersect(get_currencies(true, '', true), active_currencies('key', '', true));
        return array_diff($allowedCurrencies, [sys_settings('base_currency')]);
    }

    private function getApiUrl()
    {
        return $this->baseUrl;
    }

    private function getApiKey()
    {
        return sys_settings('exratesapi_access_key') ?: get_ex_apikey();
    }

private function getApiData()
{
    $apikey = $this->getApiKey();
    $baseCurrency = sys_settings('base_currency');

    if (!$apikey) {
        return [];
    }

    $params = [
        'access_key' => $apikey,
        'symbols' => implode(',', $this->currencies()),
    ];

    // Only add base parameter if it's EUR
    if ($baseCurrency === 'EUR') {
        $params['base'] = $baseCurrency;
    }

    return $params;
}


    private function serviceTimeout($type = null)
    {
        $output = $this->timeOut * 10;

        if (!empty($type)) {
            $service = ($type === 'invalid_access_key') ? 'no' : 'na';
            Setting::updateOrCreate(['key' => 'api_service'], ['value' => $service]);
            $output = $this->timeOut * 60;
        }

        return $output;
    }

    private function getLatestRates()
    {
        $rates = [];
        $scheduler = Cache::get('exrates_scheduler', false);
        $params = $this->getApiData();

        if (!empty($params) && empty($scheduler)) {
            try {
                $response = Http::timeout($this->timeOut)->get($this->getApiUrl(), $params);

                if ($response->successful()) {
                    $data = $response->json();
                    if (!empty($data['success']) && !empty($data['rates']) && is_array($data['rates'])) {
                        $rates = $data['rates'];

                        Setting::updateOrCreate(['key' => 'automatic_exchange_rate'], ['value' => json_encode($rates)]);
                        Setting::updateOrCreate(['key' => 'exchange_last_update'], ['value' => time()]);
                        Setting::updateOrCreate(['key' => 'exratesapi_error_msg'], ['value' => '']);
                        Setting::updateOrCreate(['key' => 'api_service'], ['value' => 'yes']);
                        Cache::forget('exrates_scheduler');
                    } else {
                        $error = $data['error'] ?? [];
                        $message = $error['info'] ?? 'Unable to fetch exchange rates.';
                        $type = $error['type'] ?? null;

                        $timeout = $this->serviceTimeout($type);
                        Log::error('exchangeratesapi-error', [$message]);

                        Setting::updateOrCreate(['key' => 'exratesapi_error_msg'], ['value' => $message]);
                        Cache::put('exrates_scheduler', time() + $timeout, $timeout);
                    }
                } else {
                    $response->throw();
                }
            } catch (\Exception $e) {
                Log::error('exchangeratesapi-exception', [$e->getMessage()]);
                Setting::updateOrCreate(['key' => 'exratesapi_error_msg'], ['value' => 'An unknown error occurred.']);
            }
        } else {
            Setting::updateOrCreate(['key' => 'exratesapi_error_msg'], ['value' => 'Missing API key or request was blocked.']);
        }

        if (empty($rates)) {
            $rates = sys_settings('automatic_exchange_rate');
        }

        return $rates;
    }

    public function refreshCache($force = false)
    {
        Cache::forget('exchange_rates');
        return $this->getExchangeRates($force);
    }

    public function getExchangeRates($force = false)
    {
        if ($force === true) {
            return $this->getLatestRates();
        }

        return Cache::remember('exchange_rates', sys_settings('exchange_auto_update', 30) * $this->timeOut, function () {
            return $this->getLatestRates();
        });
    }
}
