<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use ReflectionClass;

class ControllerLogger
{
    public function handle(Request $request, Closure $next)
    {
        // Capture Response first
        $response = $next($request);

        // Now route action is resolved
        $route = $request->route();
        $action = $route ? $route->getActionName() : null;

        $controllerClass = null;
        $controllerMethod = null;
        $controllerFile = null;

        if ($action && strpos($action, '@') !== false) {
            [$controllerClass, $controllerMethod] = explode('@', $action);

            try {
                $reflection = new ReflectionClass($controllerClass);
                $controllerFile = $reflection->getFileName();
            } catch (\ReflectionException $e) {
                $controllerFile = 'Unknown';
            }
        }

        // Request + Controller Info Log AFTER controller resolved
        Log::info('--- Incoming Request ---', [
            'Method'            => $request->method(),
            'URL'               => $request->fullUrl(),
            'ControllerClass'   => $controllerClass,
            'ControllerMethod'  => $controllerMethod,
            'ControllerFile'    => $controllerFile,
            'Payload'           => $request->all(),
            'IP'                => $request->ip(),
            'UserAgent'         => $request->userAgent(),
            'UserID'            => optional(auth()->user())->id,
        ]);

        // Response Log
        Log::info('--- Outgoing Response ---', [
            'Status'  => $response->getStatusCode(),
            'Content' => $response instanceof \Illuminate\Http\JsonResponse ? $response->getData(true) : 'Non-JSON Response',
        ]);

        return $response;
    }
}
